// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/// A filter level controlling how aggressively to filter sensitive content.
///
/// Text prompts provided as inputs and images (generated or uploaded) through Imagen on Vertex AI
/// are assessed against a list of safety filters, which include 'harmful categories' (for example,
/// `violence`, `sexual`, `derogatory`, and `toxic`). This filter level controls how aggressively to
/// filter out potentially harmful content from responses. See the
/// [`safetySetting`](https://cloud.google.com/vertex-ai/generative-ai/docs/model-reference/imagen-api#parameter_list)
/// documentation and the [Responsible AI and usage
/// guidelines](https://cloud.google.com/vertex-ai/generative-ai/docs/image/responsible-ai-imagen#safety-filters)
/// for more details.
@available(iOS 15.0, macOS 12.0, macCatalyst 15.0, tvOS 15.0, watchOS 8.0, *)
public struct ImagenSafetyFilterLevel: ProtoEnum, Sendable {
  enum Kind: String {
    case blockLowAndAbove = "block_low_and_above"
    case blockMediumAndAbove = "block_medium_and_above"
    case blockOnlyHigh = "block_only_high"
    case blockNone = "block_none"
  }

  /// The most aggressive filtering level; most strict blocking.
  public static let blockLowAndAbove = ImagenSafetyFilterLevel(kind: .blockLowAndAbove)

  /// Blocks some problematic prompts and responses.
  public static let blockMediumAndAbove = ImagenSafetyFilterLevel(kind: .blockMediumAndAbove)

  /// Reduces the number of requests blocked due to safety filters.
  ///
  /// > Important: This may increase objectionable content generated by Imagen.
  public static let blockOnlyHigh = ImagenSafetyFilterLevel(kind: .blockOnlyHigh)

  /// The least aggressive filtering level; blocks very few problematic prompts and responses.
  ///
  /// > Important: Access to this feature is restricted and may require your use case to be reviewed
  /// and approved by Cloud support.
  public static let blockNone = ImagenSafetyFilterLevel(kind: .blockNone)

  let rawValue: String
}
