/*
 * Copyright (c) Meta Platforms, Inc. and affiliates.
 * All rights reserved.
 *
 * This source code is licensed under the license found in the
 * LICENSE file in the root directory of this source tree.
 */

@testable import FBSDKCoreKit

import TestTools
import XCTest

final class InstrumentManagerTests: XCTestCase {

  var manager = _InstrumentManager.shared
  let settings = TestSettings()
  let crashObserver = TestCrashObserver()
  let errorReporter = TestErrorReporter()
  let crashHandler = TestCrashHandler()
  let featureManager = TestFeatureManager()

  override class func setUp() {
    super.setUp()
    _InstrumentManager.reset()
  }

  override func setUp() {
    super.setUp()

    manager.configure(
      featureChecker: featureManager,
      settings: settings,
      crashObserver: crashObserver,
      errorReporter: errorReporter,
      crashHandler: crashHandler
    )
  }

  override func tearDown() {
    super.tearDown()
    _InstrumentManager.reset()
  }

  func testDefaultDependencies() {
    let manager = _InstrumentManager.shared

    XCTAssertNil(
      manager.featureChecker,
      "Should not have a feature checker by default"
    )
    XCTAssertNil(
      manager.settings,
      "Should not have settings by default"
    )
    XCTAssertNil(
      manager.crashObserver,
      "Should not have a crash observer by default"
    )
    XCTAssertNil(
      manager.errorReporter,
      "Should not have an error reporter by default"
    )
    XCTAssertNil(
      manager.crashHandler,
      "Should not have a crash handler by default"
    )
  }

  func testEnablingWithBothEnabled() {
    settings.isAutoLogAppEventsEnabled = true

    manager.enable()

    featureManager.completeCheck(forFeature: .crashReport, with: true)
    featureManager.completeCheck(forFeature: .errorReport, with: true)

    XCTAssertTrue(crashHandler.wasAddObserverCalled)
    XCTAssertNotNil(crashHandler.observer)
    XCTAssertTrue(errorReporter.wasEnableCalled)
  }

  func testEnablingWithAutoLoggingEnabledAndFeaturesDisabled() {
    settings.isAutoLogAppEventsEnabled = true

    manager.enable()

    featureManager.completeCheck(forFeature: .crashReport, with: false)
    featureManager.completeCheck(forFeature: .errorReport, with: false)

    XCTAssertFalse(crashHandler.wasAddObserverCalled)
    XCTAssertNil(crashHandler.observer)
    XCTAssertFalse(errorReporter.wasEnableCalled)
  }

  func testEnablingWithAutoLoggingDisabledAndFeaturesEnabled() {
    settings.isAutoLogAppEventsEnabled = false

    manager.enable()

    featureManager.completeCheck(forFeature: .crashReport, with: true)
    featureManager.completeCheck(forFeature: .errorReport, with: true)

    XCTAssertFalse(crashHandler.wasAddObserverCalled)
    XCTAssertNil(crashHandler.observer)
    XCTAssertFalse(errorReporter.wasEnableCalled)
  }

  func testEnablingWithBothDisabled() {
    settings.isAutoLogAppEventsEnabled = false

    manager.enable()

    featureManager.completeCheck(forFeature: .crashReport, with: false)
    featureManager.completeCheck(forFeature: .errorReport, with: false)

    XCTAssertFalse(crashHandler.wasAddObserverCalled)
    XCTAssertNil(crashHandler.observer)
    XCTAssertFalse(errorReporter.wasEnableCalled)
  }
}
