package com.bingougame.sdk;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.DialogInterface;

import com.bingougame.sdk.ResourceUtil;
import com.bingougame.sdk.permission.PermissionConstants;
import com.bingougame.sdk.permission.PermissionLang;
import com.bingougame.sdk.permission.PermissionUtils;
import com.bingougame.sdk.utils.ActivityUtils;

import java.util.List;

/**
 * <pre>
 *     author: Blankj
 *     blog  : http://blankj.comc# app.config
 *     time  : 2018/01/06
 *     desc  : 权限帮助类
 * </pre>
 */
public class PermissionHelper {

    public static void requestMicroPhone(boolean enforce,
                                         final OnPermissionGrantedListener grantedListener,
                                         final OnPermissionDeniedListener deniedListener) {
        PermissionUtils.permission(PermissionConstants.MICROPHONE)
                .callback(new PermissionUtils.SimpleCallback() {
                    @Override
                    public void onGranted() {
                        grantedListener.onPermissionGranted();
                    }

                    @Override
                    public void onDenied() {
                        deniedListener.onPermissionDenied(true);
                    }
                })
                .request();
    }

    /**
     * 权限申请具体执行方法
     * @param enforce          是否强制申请权限，就是该权限控制的功能是否是游戏运行的必须功能
     * @param grantedListener  授予权限的回调监听器
     * @param deniedListener   拒绝权限的回调监听器
     * @param permissions      申请权限的名称
     */
    private static void request(final boolean enforce,
                                final PermissionLang lang,
                                final OnPermissionGrantedListener grantedListener,
                                final OnPermissionDeniedListener deniedListener,
                                final String[] enforcePermissions,
                                final @PermissionConstants.Permission String... permissions) {
        //获取需要PermissionUtils对象,并构造对象属性
        PermissionUtils.permission(permissions)
                .callback(new PermissionUtils.FullCallback() {
                    @Override
                    public void onGranted(List<String> permissionsGranted) {
                        grantedListener.onPermissionGranted();
                    }
                    @Override
                    public void onDenied(List<String> permissionsDeniedForever, List<String> permissionsDenied) {
                        //如果申请存储权限被用户拒绝,那么就展示游戏无法正常启动原因的对话框
                        if (checkEnforcePermission(permissionsDeniedForever,enforcePermissions)) {
                            showOpenAppSettingDialog(enforce,lang,deniedListener);
                        }else if (checkEnforcePermission(permissionsDenied,enforcePermissions)){
                            PermissionHelper.showDeniedDialog(enforce,lang,grantedListener,deniedListener,enforcePermissions);
                        } else {
                            deniedListener.onPermissionDenied(true);
                        }
                    }
                })
                .request();
    }

    private static boolean checkEnforcePermission(List<String> permissionDenied, String[] enforcePermissions){
        if (permissionDenied == null ||
                enforcePermissions == null ||
                enforcePermissions.length == 0){
            return false;
        }
        for (String permission : enforcePermissions) {
            for (String sub : PermissionConstants.getPermissions(permission)){
                if (permissionDenied.contains(sub)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static boolean checkDangerousPermission() {
        return PermissionUtils.isGranted(PermissionConstants.STORAGE,
                PermissionConstants.SMS,
                PermissionConstants.SENSORS,
                PermissionConstants.PHONE,
//                PermissionConstants.MICROPHONE,
                PermissionConstants.LOCATION,
                PermissionConstants.CONTACTS,
                PermissionConstants.CAMERA,
                PermissionConstants.CALENDAR);
    }

    public static void requestDangerousPermission(boolean enforce,String[] enforcePermissions, OnPermissionGrantedListener grantedListener, OnPermissionDeniedListener deniedListener) {
        Activity topActivity = ActivityUtils.getTopActivity();
        if (topActivity == null) {
            return;
        }
        final PermissionLang lang = PermissionLang.getInstance(topActivity,enforcePermissions);
        request(enforce,lang,grantedListener,deniedListener,enforcePermissions,
                PermissionConstants.STORAGE,
                PermissionConstants.SMS,
                PermissionConstants.SENSORS,
                PermissionConstants.PHONE,
//                PermissionConstants.MICROPHONE,
                PermissionConstants.LOCATION,
                PermissionConstants.CONTACTS,
                PermissionConstants.CAMERA,
                PermissionConstants.CALENDAR);
    }

    public interface OnPermissionGrantedListener {
        void onPermissionGranted();
    }

    public interface OnPermissionDeniedListener {
        void onPermissionDenied(boolean result);
    }

    /**
     * 权限申请被拒的提示对话框
     * @param enforce          是否强制申请权限，就是该权限控制的功能是否是游戏运行的必须功能
     * @param grantedListener  授予权限的回调监听器
     * @param deniedListener   拒绝权限的回调监听器
     * @param permissions      申请权限的名称
     */
    public static void showDeniedDialog(final boolean enforce,
                                           final PermissionLang lang,
                                           final OnPermissionGrantedListener grantedListener,
                                           final OnPermissionDeniedListener deniedListener,
                                           final @PermissionConstants.Permission String... permissions){
        Activity topActivity = ActivityUtils.getTopActivity();
        if (topActivity == null) {
            return;
        }

        AlertDialog.Builder builder = new AlertDialog.Builder(topActivity);
        builder.setTitle(lang.getTitleText())
                .setMessage(lang.getDeniedMsg())
                .setPositiveButton(lang.getPermissionRequestText(), new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        request(enforce,lang,grantedListener,deniedListener,permissions,permissions);
                    }
                })
                .setCancelable(false);
        //如果是强制权限申请，不给权限就只能退出游戏
        if(enforce){
            builder.setNegativeButton(lang.getExitGameText(), new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    System.exit(0);
                }
            });
        }else{
            builder.setNegativeButton(lang.getConfirmText(), new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    dialog.dismiss();
                    deniedListener.onPermissionDenied(true);
                }
            });
        }
        builder.create().show();
    }

    /**
     * 永久拒绝权限且必须开启的权限,请求用户去设置中开启
     */
    public static void showOpenAppSettingDialog(boolean enforce,PermissionLang lang, final OnPermissionDeniedListener deniedListener) {
        Activity topActivity = ActivityUtils.getTopActivity();
        if (topActivity == null) {
            return;
        }
        AlertDialog.Builder builder = new AlertDialog.Builder(topActivity);
        builder.setTitle(lang.getTitleText())
                .setMessage(lang.getDeniedForeverMsg())
                .setPositiveButton(lang.getSettingText(), new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        deniedListener.onPermissionDenied(false);
                        PermissionUtils.openAppSettings();
                    }
                })
                .setCancelable(false);

        if (enforce){
            builder.setNegativeButton(lang.getExitGameText(), new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                        System.exit(0);
                }
            });
        }else {
            builder.setNegativeButton(lang.getConfirmText(), new DialogInterface.OnClickListener() {
                @Override
                public void onClick(DialogInterface dialog, int which) {
                    dialog.dismiss();
                    deniedListener.onPermissionDenied(true);
                }
            });
        }
        builder.create().show();
    }



}
