/*
 * Copyright (c) 2009, 2019, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */


package org.graalvm.compiler.nodes.calc;

import org.graalvm.compiler.core.common.type.AbstractPointerStamp;
import org.graalvm.compiler.core.common.type.ObjectStamp;
import org.graalvm.compiler.core.common.type.Stamp;
import org.graalvm.compiler.graph.NodeClass;
import org.graalvm.compiler.graph.spi.CanonicalizerTool;
import org.graalvm.compiler.nodeinfo.NodeCycles;
import org.graalvm.compiler.nodeinfo.NodeInfo;
import org.graalvm.compiler.nodes.CompressionNode;
import org.graalvm.compiler.nodes.LogicConstantNode;
import org.graalvm.compiler.nodes.LogicNode;
import org.graalvm.compiler.nodes.NodeView;
import org.graalvm.compiler.nodes.UnaryOpLogicNode;
import org.graalvm.compiler.nodes.ValueNode;
import org.graalvm.compiler.nodes.spi.LIRLowerable;
import org.graalvm.compiler.nodes.spi.NodeLIRBuilderTool;
import org.graalvm.compiler.nodes.spi.Virtualizable;
import org.graalvm.compiler.nodes.spi.VirtualizerTool;
import org.graalvm.compiler.nodes.type.NarrowOopStamp;
import org.graalvm.compiler.nodes.type.StampTool;

import jdk.vm.ci.meta.JavaConstant;
import jdk.vm.ci.meta.TriState;

/**
 * An IsNullNode will be true if the supplied value is null, and false if it is non-null.
 */
@NodeInfo(cycles = NodeCycles.CYCLES_2)
public final class IsNullNode extends UnaryOpLogicNode implements LIRLowerable, Virtualizable {

    public static final NodeClass<IsNullNode> TYPE = NodeClass.create(IsNullNode.class);

    /*
     * When linear pointer compression is enabled, compressed and uncompressed nulls differ.
     */
    private final JavaConstant nullConstant;

    private IsNullNode(ValueNode object, JavaConstant nullConstant) {
        super(TYPE, object);
        this.nullConstant = nullConstant;
        assert object != null;
    }

    public IsNullNode(ValueNode object) {
        this(object, JavaConstant.NULL_POINTER);
        assertNonNarrow(object);
    }

    public JavaConstant nullConstant() {
        return nullConstant;
    }

    public static LogicNode create(ValueNode forValue) {
        assertNonNarrow(forValue);
        return canonicalized(null, forValue, JavaConstant.NULL_POINTER);
    }

    public static LogicNode create(ValueNode forValue, JavaConstant nullConstant) {
        assert nullConstant.isNull() : "Null constant is not null: " + nullConstant;
        return canonicalized(null, forValue, nullConstant);
    }

    private static void assertNonNarrow(ValueNode object) {
        assert !(object.stamp(NodeView.DEFAULT) instanceof NarrowOopStamp) : "Value to compare against null is a NarrowOop" + object;
    }

    @Override
    public void generate(NodeLIRBuilderTool gen) {
        // Nothing to do.
    }

    @Override
    public boolean verify() {
        assertTrue(getValue() != null, "is null input must not be null");
        assertTrue(getValue().stamp(NodeView.DEFAULT).isPointerStamp(), "input must be a pointer not %s", getValue().stamp(NodeView.DEFAULT));
        return super.verify();
    }

    @Override
    public ValueNode canonical(CanonicalizerTool tool, ValueNode forValue) {
        return canonicalized(this, forValue, nullConstant);
    }

    private static LogicNode canonicalized(IsNullNode node, ValueNode forValue, JavaConstant forNullConstant) {
        JavaConstant nullConstant = forNullConstant;
        ValueNode value = forValue;
        while (true) {
            if (StampTool.isPointerAlwaysNull(value)) {
                return LogicConstantNode.tautology();
            } else if (StampTool.isPointerNonNull(value)) {
                return LogicConstantNode.contradiction();
            }

            if (value instanceof ConvertNode) {
                ConvertNode convertNode = (ConvertNode) value;
                if (convertNode.mayNullCheckSkipConversion()) {
                    value = convertNode.getValue();
                    continue;
                }
                /*
                 * CompressionNode.mayNullCheckSkipConversion returns false when linear pointer
                 * compression is enabled.
                 */
                if (value instanceof CompressionNode) {
                    CompressionNode compressionNode = (CompressionNode) value;
                    nullConstant = compressionNode.nullConstant();
                    value = compressionNode.getValue();
                    continue;
                }
            }
            /*
             * If we are at original node, just return it. Otherwise create a new node.
             */
            return (node != null && value == forValue) ? node : new IsNullNode(value, nullConstant);
        }
    }

    @Override
    public void virtualize(VirtualizerTool tool) {
        ValueNode alias = tool.getAlias(getValue());
        TriState fold = tryFold(alias.stamp(NodeView.DEFAULT));
        if (fold != TriState.UNKNOWN) {
            tool.replaceWithValue(LogicConstantNode.forBoolean(fold.isTrue(), graph()));
        }
    }

    @Override
    public Stamp getSucceedingStampForValue(boolean negated) {
        // Ignore any more precise input stamp since canonicalization will skip through PiNodes
        AbstractPointerStamp pointerStamp = (AbstractPointerStamp) getValue().stamp(NodeView.DEFAULT).unrestricted();
        return negated ? pointerStamp.asNonNull() : pointerStamp.asAlwaysNull();
    }

    @Override
    public TriState tryFold(Stamp valueStamp) {
        if (valueStamp instanceof ObjectStamp) {
            ObjectStamp objectStamp = (ObjectStamp) valueStamp;
            if (objectStamp.alwaysNull()) {
                return TriState.TRUE;
            } else if (objectStamp.nonNull()) {
                return TriState.FALSE;
            }
        }
        return TriState.UNKNOWN;
    }
}
