/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.math;

import com.google.common.annotations.Beta;
import com.google.common.annotations.GwtCompatible;
import com.google.common.annotations.GwtIncompatible;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.math.LongMath;
import com.google.common.math.MathPreconditions;
import com.google.common.primitives.Ints;
import java.math.RoundingMode;

@GwtCompatible(emulated=true)
public final class IntMath {
    @VisibleForTesting
    static final int MAX_SIGNED_POWER_OF_TWO = 0x40000000;
    @VisibleForTesting
    static final int MAX_POWER_OF_SQRT2_UNSIGNED = -1257966797;
    @VisibleForTesting
    static final byte[] maxLog10ForLeadingZeros = new byte[]{9, 9, 9, 8, 8, 8, 7, 7, 7, 6, 6, 6, 6, 5, 5, 5, 4, 4, 4, 3, 3, 3, 3, 2, 2, 2, 1, 1, 1, 0, 0, 0, 0};
    @VisibleForTesting
    static final int[] powersOf10 = new int[]{1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000};
    @VisibleForTesting
    static final int[] halfPowersOf10 = new int[]{3, 31, 316, 3162, 31622, 316227, 3162277, 31622776, 316227766, Integer.MAX_VALUE};
    @VisibleForTesting
    static final int FLOOR_SQRT_MAX_INT = 46340;
    private static final int[] factorials = new int[]{1, 1, 2, 6, 24, 120, 720, 5040, 40320, 362880, 3628800, 39916800, 479001600};
    @VisibleForTesting
    static int[] biggestBinomials = new int[]{Integer.MAX_VALUE, Integer.MAX_VALUE, 65536, 2345, 477, 193, 110, 75, 58, 49, 43, 39, 37, 35, 34, 34, 33};

    @Beta
    public static int ceilingPowerOfTwo(int x4) {
        MathPreconditions.checkPositive("x", x4);
        if (x4 > 0x40000000) {
            throw new ArithmeticException(new StringBuilder(58).append("ceilingPowerOfTwo(").append(x4).append(") not representable as an int").toString());
        }
        return 1 << -Integer.numberOfLeadingZeros(x4 - 1);
    }

    @Beta
    public static int floorPowerOfTwo(int x4) {
        MathPreconditions.checkPositive("x", x4);
        return Integer.highestOneBit(x4);
    }

    public static boolean isPowerOfTwo(int x4) {
        return x4 > 0 & (x4 & x4 - 1) == 0;
    }

    @VisibleForTesting
    static int lessThanBranchFree(int x4, int y4) {
        return ~(~(x4 - y4)) >>> 31;
    }

    public static int log2(int x4, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x4);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(IntMath.isPowerOfTwo(x4));
            }
            case DOWN: 
            case FLOOR: {
                return 31 - Integer.numberOfLeadingZeros(x4);
            }
            case UP: 
            case CEILING: {
                return 32 - Integer.numberOfLeadingZeros(x4 - 1);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int leadingZeros = Integer.numberOfLeadingZeros(x4);
                int cmp = -1257966797 >>> leadingZeros;
                int logFloor = 31 - leadingZeros;
                return logFloor + IntMath.lessThanBranchFree(cmp, x4);
            }
        }
        throw new AssertionError();
    }

    @GwtIncompatible
    public static int log10(int x4, RoundingMode mode) {
        MathPreconditions.checkPositive("x", x4);
        int logFloor = IntMath.log10Floor(x4);
        int floorPow = powersOf10[logFloor];
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(x4 == floorPow);
            }
            case DOWN: 
            case FLOOR: {
                return logFloor;
            }
            case UP: 
            case CEILING: {
                return logFloor + IntMath.lessThanBranchFree(floorPow, x4);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                return logFloor + IntMath.lessThanBranchFree(halfPowersOf10[logFloor], x4);
            }
        }
        throw new AssertionError();
    }

    private static int log10Floor(int x4) {
        byte y4 = maxLog10ForLeadingZeros[Integer.numberOfLeadingZeros(x4)];
        return y4 - IntMath.lessThanBranchFree(x4, powersOf10[y4]);
    }

    @GwtIncompatible
    public static int pow(int b4, int k4) {
        MathPreconditions.checkNonNegative("exponent", k4);
        switch (b4) {
            case 0: {
                return k4 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k4 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                return k4 < 32 ? 1 << k4 : 0;
            }
            case -2: {
                if (k4 < 32) {
                    return (k4 & 1) == 0 ? 1 << k4 : -(1 << k4);
                }
                return 0;
            }
        }
        int accum = 1;
        while (true) {
            switch (k4) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return b4 * accum;
                }
            }
            accum *= (k4 & 1) == 0 ? 1 : b4;
            b4 *= b4;
            k4 >>= 1;
        }
    }

    @GwtIncompatible
    public static int sqrt(int x4, RoundingMode mode) {
        MathPreconditions.checkNonNegative("x", x4);
        int sqrtFloor = IntMath.sqrtFloor(x4);
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(sqrtFloor * sqrtFloor == x4);
            }
            case DOWN: 
            case FLOOR: {
                return sqrtFloor;
            }
            case UP: 
            case CEILING: {
                return sqrtFloor + IntMath.lessThanBranchFree(sqrtFloor * sqrtFloor, x4);
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int halfSquare = sqrtFloor * sqrtFloor + sqrtFloor;
                return sqrtFloor + IntMath.lessThanBranchFree(halfSquare, x4);
            }
        }
        throw new AssertionError();
    }

    private static int sqrtFloor(int x4) {
        return (int)Math.sqrt(x4);
    }

    public static int divide(int p4, int q4, RoundingMode mode) {
        boolean increment;
        Preconditions.checkNotNull(mode);
        if (q4 == 0) {
            throw new ArithmeticException("/ by zero");
        }
        int div = p4 / q4;
        int rem = p4 - q4 * div;
        if (rem == 0) {
            return div;
        }
        int signum = 1 | (p4 ^ q4) >> 31;
        switch (mode) {
            case UNNECESSARY: {
                MathPreconditions.checkRoundingUnnecessary(rem == 0);
            }
            case DOWN: {
                increment = false;
                break;
            }
            case UP: {
                increment = true;
                break;
            }
            case CEILING: {
                increment = signum > 0;
                break;
            }
            case FLOOR: {
                increment = signum < 0;
                break;
            }
            case HALF_DOWN: 
            case HALF_UP: 
            case HALF_EVEN: {
                int absRem = Math.abs(rem);
                int cmpRemToHalfDivisor = absRem - (Math.abs(q4) - absRem);
                if (cmpRemToHalfDivisor == 0) {
                    increment = mode == RoundingMode.HALF_UP || mode == RoundingMode.HALF_EVEN & (div & 1) != 0;
                    break;
                }
                increment = cmpRemToHalfDivisor > 0;
                break;
            }
            default: {
                throw new AssertionError();
            }
        }
        return increment ? div + signum : div;
    }

    public static int mod(int x4, int m4) {
        if (m4 <= 0) {
            throw new ArithmeticException(new StringBuilder(31).append("Modulus ").append(m4).append(" must be > 0").toString());
        }
        int result = x4 % m4;
        return result >= 0 ? result : result + m4;
    }

    public static int gcd(int a5, int b4) {
        MathPreconditions.checkNonNegative("a", a5);
        MathPreconditions.checkNonNegative("b", b4);
        if (a5 == 0) {
            return b4;
        }
        if (b4 == 0) {
            return a5;
        }
        int aTwos = Integer.numberOfTrailingZeros(a5);
        a5 >>= aTwos;
        int bTwos = Integer.numberOfTrailingZeros(b4);
        b4 >>= bTwos;
        while (a5 != b4) {
            int delta = a5 - b4;
            int minDeltaOrZero = delta & delta >> 31;
            a5 = delta - minDeltaOrZero - minDeltaOrZero;
            b4 += minDeltaOrZero;
            a5 >>= Integer.numberOfTrailingZeros(a5);
        }
        return a5 << Math.min(aTwos, bTwos);
    }

    public static int checkedAdd(int a5, int b4) {
        long result = (long)a5 + (long)b4;
        MathPreconditions.checkNoOverflow(result == (long)((int)result), "checkedAdd", a5, b4);
        return (int)result;
    }

    public static int checkedSubtract(int a5, int b4) {
        long result = (long)a5 - (long)b4;
        MathPreconditions.checkNoOverflow(result == (long)((int)result), "checkedSubtract", a5, b4);
        return (int)result;
    }

    public static int checkedMultiply(int a5, int b4) {
        long result = (long)a5 * (long)b4;
        MathPreconditions.checkNoOverflow(result == (long)((int)result), "checkedMultiply", a5, b4);
        return (int)result;
    }

    public static int checkedPow(int b4, int k4) {
        MathPreconditions.checkNonNegative("exponent", k4);
        switch (b4) {
            case 0: {
                return k4 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k4 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                MathPreconditions.checkNoOverflow(k4 < 31, "checkedPow", b4, k4);
                return 1 << k4;
            }
            case -2: {
                MathPreconditions.checkNoOverflow(k4 < 32, "checkedPow", b4, k4);
                return (k4 & 1) == 0 ? 1 << k4 : -1 << k4;
            }
        }
        int accum = 1;
        while (true) {
            switch (k4) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return IntMath.checkedMultiply(accum, b4);
                }
            }
            if ((k4 & 1) != 0) {
                accum = IntMath.checkedMultiply(accum, b4);
            }
            if ((k4 >>= 1) <= 0) continue;
            MathPreconditions.checkNoOverflow(-46340 <= b4 & b4 <= 46340, "checkedPow", b4, k4);
            b4 *= b4;
        }
    }

    @Beta
    public static int saturatedAdd(int a5, int b4) {
        return Ints.saturatedCast((long)a5 + (long)b4);
    }

    @Beta
    public static int saturatedSubtract(int a5, int b4) {
        return Ints.saturatedCast((long)a5 - (long)b4);
    }

    @Beta
    public static int saturatedMultiply(int a5, int b4) {
        return Ints.saturatedCast((long)a5 * (long)b4);
    }

    @Beta
    public static int saturatedPow(int b4, int k4) {
        MathPreconditions.checkNonNegative("exponent", k4);
        switch (b4) {
            case 0: {
                return k4 == 0 ? 1 : 0;
            }
            case 1: {
                return 1;
            }
            case -1: {
                return (k4 & 1) == 0 ? 1 : -1;
            }
            case 2: {
                if (k4 >= 31) {
                    return Integer.MAX_VALUE;
                }
                return 1 << k4;
            }
            case -2: {
                if (k4 >= 32) {
                    return Integer.MAX_VALUE + (k4 & 1);
                }
                return (k4 & 1) == 0 ? 1 << k4 : -1 << k4;
            }
        }
        int accum = 1;
        int limit = Integer.MAX_VALUE + (b4 >>> 31 & (k4 & 1));
        while (true) {
            switch (k4) {
                case 0: {
                    return accum;
                }
                case 1: {
                    return IntMath.saturatedMultiply(accum, b4);
                }
            }
            if ((k4 & 1) != 0) {
                accum = IntMath.saturatedMultiply(accum, b4);
            }
            if ((k4 >>= 1) <= 0) continue;
            if (-46340 > b4 | b4 > 46340) {
                return limit;
            }
            b4 *= b4;
        }
    }

    public static int factorial(int n4) {
        MathPreconditions.checkNonNegative("n", n4);
        return n4 < factorials.length ? factorials[n4] : Integer.MAX_VALUE;
    }

    public static int binomial(int n4, int k4) {
        MathPreconditions.checkNonNegative("n", n4);
        MathPreconditions.checkNonNegative("k", k4);
        Preconditions.checkArgument(k4 <= n4, "k (%s) > n (%s)", k4, n4);
        if (k4 > n4 >> 1) {
            k4 = n4 - k4;
        }
        if (k4 >= biggestBinomials.length || n4 > biggestBinomials[k4]) {
            return Integer.MAX_VALUE;
        }
        switch (k4) {
            case 0: {
                return 1;
            }
            case 1: {
                return n4;
            }
        }
        long result = 1L;
        for (int i4 = 0; i4 < k4; ++i4) {
            result *= (long)(n4 - i4);
            result /= (long)(i4 + 1);
        }
        return (int)result;
    }

    public static int mean(int x4, int y4) {
        return (x4 & y4) + ((x4 ^ y4) >> 1);
    }

    @GwtIncompatible
    @Beta
    public static boolean isPrime(int n4) {
        return LongMath.isPrime(n4);
    }

    private IntMath() {
    }
}

